/*
 * SchemeSelectorDialog.java - for Editor Scheme Plugin
 * Copyright (C) 2000 Ollie Rutherfurd
 *
 * :folding=explicit:collapseFolds=1:
 *
 * {{{This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.}}}
 *
 * $Id: EditorSchemeSelectorDialog.java,v 1.8 2003/11/10 14:23:19 orutherfurd Exp $
 */

package editorscheme;

//{{{ imports
import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import org.gjt.sp.jedit.gui.*;
import org.gjt.sp.jedit.*;

import org.gjt.sp.util.Log;
//}}}

/**
* Displays available jEdit 'Editor Schemes' and allows the user to switch
* between schemes and update writable schemes.
*/
public class EditorSchemeSelectorDialog extends EnhancedDialog implements ActionListener
{

	private EditorScheme selectedScheme;
	private EditorScheme original;
	private JList schemeList;
	Vector schemes;
	private JCheckBox autoApply;
	private JCheckBoxList groupsList;
	private JButton closeButton;
	private JButton newButton;
	private JButton saveButton;
	private JButton selectButton;

	//{{{ constructor
	public EditorSchemeSelectorDialog(View view)
	{
		super(view, jEdit.getProperty("editor-scheme.selector.title"), true);

		JPanel schemesPanel = new JPanel(new BorderLayout());
		JPanel main = new JPanel(new GridLayout(1,2,5,5));

		JPanel content = new JPanel(new BorderLayout());
		content.setBorder(new EmptyBorder(12,12,12,12));
		setContentPane(content);

		// get the list of schemes
		EditorSchemePlugin.loadSchemes();
		schemes = new Vector(EditorSchemePlugin.getSchemes());

		// get current settings, but don't add until a new scheme has been applied
		original = new EditorScheme();
		original.getFromCurrent();
		original.setName(jEdit.getProperty("editor-scheme.selector.currentscheme"));
		schemes.insertElementAt(original,0);

		schemeList = new JList(schemes);
		schemeList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		schemeList.setCellRenderer(new EditorSchemeListCellRenderer());
		ListSelectionHandler listSelectionHandler = new ListSelectionHandler();
		schemeList.addListSelectionListener(listSelectionHandler);
		schemeList.addMouseListener(listSelectionHandler);
		schemesPanel.add(
			new JLabel(jEdit.getProperty("editor-scheme.schemes")),
			BorderLayout.NORTH);
		schemesPanel.add(new JScrollPane(schemeList), BorderLayout.CENTER);
		autoApply = new JCheckBox(
			jEdit.getProperty("editor-scheme.autoapply.label"),
			jEdit.getBooleanProperty("editor-scheme.autoapply", true));
		schemesPanel.add(autoApply, BorderLayout.SOUTH);
		main.add(schemesPanel);

		JPanel groupsPanel = new JPanel(new BorderLayout());
		Vector groups = EditorScheme.getPropertyGroups();
		JCheckBoxList.Entry[] entries = new JCheckBoxList.Entry[groups.size()];
		for(int i=0; i < groups.size(); i++)
		{
			EditorScheme.PropertyGroup group = 
				(EditorScheme.PropertyGroup)groups.elementAt(i);
			entries[i] = new JCheckBoxList.Entry(group.apply, group);
		}
		groupsList = new JCheckBoxList(entries);
		groupsPanel.add(
			new JLabel(jEdit.getProperty("editor-scheme.usegroups")), 
			BorderLayout.NORTH);
		groupsPanel.add(new JScrollPane(groupsList), BorderLayout.CENTER);

		main.add(groupsPanel);
		content.add(main, BorderLayout.CENTER);

		JPanel buttonsPanel = new JPanel();
		buttonsPanel.setBorder(new EmptyBorder(12,0,0,0));
		buttonsPanel.setLayout(new BoxLayout(buttonsPanel, BoxLayout.X_AXIS));
		buttonsPanel.add(Box.createGlue());

		selectButton = new JButton(jEdit.getProperty("editor-scheme.select"));
		selectButton.setEnabled(false);
		selectButton.addActionListener(this);
		buttonsPanel.add(selectButton);
		buttonsPanel.add(Box.createHorizontalStrut(6));
		getRootPane().setDefaultButton(selectButton);

		newButton = new JButton(jEdit.getProperty("editor-scheme.new"));
		newButton.addActionListener(this);
		buttonsPanel.add(newButton);
		buttonsPanel.add(Box.createHorizontalStrut(6));

		saveButton = new JButton(jEdit.getProperty("editor-scheme.update"));
		saveButton.setEnabled(false);
		saveButton.addActionListener(this);
		buttonsPanel.add(saveButton);
		buttonsPanel.add(Box.createHorizontalStrut(6));

		closeButton = new JButton(jEdit.getProperty("editor-scheme.close"));
		closeButton.addActionListener(this);
		buttonsPanel.add(closeButton);

		buttonsPanel.add(Box.createGlue());

		content.add(buttonsPanel, BorderLayout.SOUTH);

		pack();
		setLocationRelativeTo(view);
		show();
	}//}}}

	//{{{ getPreferredSize
	public Dimension getPreferredSize()
	{
		return new Dimension(400,300);
	}//}}}

	//{{{ Enhanced Dialog implementation
	public void ok(){}

	public void cancel()
	{
		dispose();
	}
	//}}}

	//{{{ selectScheme()
	public void selectScheme()
	{
		EditorScheme scheme = (EditorScheme)schemeList.getSelectedValue();

		Object[] entries  = groupsList.getCheckedValues();
		for(int i=0; i < entries.length; i++)
		{
			EditorScheme.PropertyGroup group = 
				(EditorScheme.PropertyGroup)entries[i];
			group.apply(scheme);
		}

		jEdit.propertiesChanged();
		jEdit.saveSettings();
	}//}}}

	//{{{ saveScheme()
	/**
	 * Updates the currently selected scheme with the properties from 
	 * '[Current Settings]', which may not be the current settings if
	 * the user has updated the settings.
	 */
	public void saveScheme()
	{
		String args[] = new String[]{  selectedScheme.getName() };
		int response = GUIUtilities.confirm(this
			,"editor-scheme.confirm-update"
			,args,JOptionPane.YES_NO_OPTION
			,JOptionPane.QUESTION_MESSAGE);

		if(response == JOptionPane.NO_OPTION)
			return;

		try
		{
			selectedScheme.getFromCurrent();
			selectedScheme.save();
			if(autoApply.isSelected())
				selectScheme();
		}
		catch(IOException ioe)
		{
			GUIUtilities.error(this,"editor-scheme.saveerror",
				new String[]{ selectedScheme.getName(), ioe.toString(), });
		}

	}//}}}


	void newScheme()
	{
		String name = "";
		String fileName = "";
		try
		{
			name = GUIUtilities.input(this, 
				"editor-scheme.save-current",  "");
			if(name == null)
				return;

			fileName = MiscUtilities.constructPath(
				EditorScheme.getDefaultDir(), 
				name.replace(' ', '_') + EditorScheme.EXTENSION);

			File file = new File(fileName);
			if(file.exists())
			{
				String[] args = new String[]{ file.getPath() };
				GUIUtilities.error(this, "editor-scheme.schemeexists", args);
				return;
			}

			EditorScheme scheme = new EditorScheme();
			scheme.setName(name);
			scheme.setFilename(fileName);
			scheme.getFromCurrent();
			scheme.setReadOnly(false);
			scheme.save();

			int i;
			for(i = 0; i < schemes.size(); i++)
			{
				EditorScheme s = (EditorScheme)schemes.elementAt(i);
				if(MiscUtilities.compareStrings(s.getName(), name , true) > 1)
					break;
			}
			schemes.insertElementAt(scheme,i);
			schemeList.setListData(schemes);

		}
		catch(IOException ioe)
		{
			String[] args = new String[]{ name, ioe.toString(), };

			GUIUtilities.error(this,"editor-scheme.saveerror",args);

			Log.log(Log.ERROR, EditorSchemePlugin.class,
				"error saving [" + name + "] to ["
				+ fileName + "]: " + ioe.toString());
		}
	}

	//{{{ ActionListener implementation
	public void actionPerformed(ActionEvent evt)
	{
		Object source = evt.getSource();

		if(source == closeButton)
		{
			cancel();
		}
		else if(source == newButton)
		{
			newScheme();
		}
		else if(source == selectButton)
		{
			selectScheme();
		}
		if(source == saveButton)
		{
			saveScheme();
		}
	}//}}}


	//{{{ ListSelectionHandler
	class ListSelectionHandler extends MouseAdapter 
		implements ListSelectionListener
	{

		public void mouseClicked(MouseEvent e)
		{
			if(e.getClickCount() == 2)
				selectScheme();
		}

		public void valueChanged(ListSelectionEvent evt)
		{
			Object selected = schemeList.getSelectedValue();
			if(selected instanceof EditorScheme)
			{
				selectedScheme = (EditorScheme) selected;
				saveButton.setEnabled(selectedScheme.getReadOnly() == false);
				selectButton.setEnabled(true);
				if(autoApply.isSelected())
					selectScheme();
			}
			else
			{
				selectButton.setEnabled(false);
				saveButton.setEnabled(false);
			}
		}
	}//}}}

	//{{{ EditorSchemeListCellRenderer class
	static class EditorSchemeListCellRenderer extends JLabel implements ListCellRenderer
	{
		EditorSchemeListCellRenderer()
		{
			setOpaque(true);
		}
	
		private ImageIcon readOnlyIcon = new
			ImageIcon(getClass().getResource("/icons/readonly.gif"));
		private ImageIcon normalIcon = new
			ImageIcon(getClass().getResource("/icons/normal.gif"));
	
		public Component getListCellRendererComponent(
			JList list,
			Object value,			// value to display
			int index,				// cell index
			boolean isSelected,		// is the cell selected
			boolean cellHasFocus)	// the list and the cell have the focus
		{
			EditorScheme scheme = (EditorScheme)value;
			String s = value.toString();
			setText(s);
	
			if(scheme.getReadOnly())
			{
				setIcon(readOnlyIcon);
			}
			else
			{
				setIcon(normalIcon);
			}
	
			if (isSelected)
			{
				setBackground(list.getSelectionBackground());
				setForeground(list.getSelectionForeground());
			}
			else
			{
				setBackground(list.getBackground());
				setForeground(list.getForeground());
			}
	
			setEnabled(list.isEnabled());
			setFont(list.getFont());
	
			return this;
		}
	}//}}}

}
