/*
* 20:47:04 04/06/00
*
* XTree.java - A tree used for XInsert system
* Original Version Copyright (C) 1999 Romain Guy - guy.romain@bigfoot.com
* Potion copyright (C) 2000 Richard Lowe
* Copyright (C) 2000 Dominic Stolerman - dominic@sspd.org.uk
* www.chez.com/powerteam/jext
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

import java.io.*;
import java.util.*;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;

import org.gjt.sp.jedit.*;
import org.gjt.sp.util.Log;
import org.gjt.sp.jedit.textarea.*;

public class XTree extends JPanel implements  TreeSelectionListener,  ActionListener
{
  private XTreeTree tree;
  private View parent;
  private static Vector inserts;
  private DefaultTreeModel treeModel;
  private JButton expand, collapse, reload;
  private JCheckBox carriageReturn, executeScript;
  private boolean treeJustCollapsed = false;

  // nested submenus
  private int rootIndex;
  private XTreeNode root;
  private Stack menuStack = null;
  private XTreeObject xtreeObj = null;

  //private int clicks;
  private int lineNo;

  public void addMenu(String nodeName)
  {
    xtreeObj = new XTreeObject(new XTreeNode(nodeName), 0);

    if (menuStack.empty())
    {
      treeModel.insertNodeInto(xtreeObj.getXTreeNode(), root, rootIndex);
      rootIndex++;
    }
    else
    {
      //Log.log(Log.DEBUG, this, "Adding menu on mennu stack: " + nodeName);
      XTreeObject obj = (XTreeObject) menuStack.peek();
      treeModel.insertNodeInto(xtreeObj.getXTreeNode(), obj.getXTreeNode(), obj.getIndex());
      obj.incrementIndex();
    }

    menuStack.push(xtreeObj);
  }

  public void closeMenu()
  {
    try
    {
      xtreeObj = (XTreeObject) menuStack.pop();
    }
    catch (Exception e)
    {
      xtreeObj = null;
    }
  }

  public void addVariable(String key, String value)
  {
    XTreeObject obj = (XTreeObject)menuStack.peek();
    XTreeNode node;
    if (obj != null)
      node = obj.getXTreeNode();
    else
      node = root;
    node.addVariable(key, value);
  }

  public void addInsert(String nodeName, String content, int script)
  {
    inserts.addElement(new XTreeItem(content, script));
    XTreeNode node = new XTreeNode(nodeName, inserts.size());

    if (xtreeObj == null)
    {
      treeModel.insertNodeInto(node, root, rootIndex);
      ++rootIndex;
    }
    else
    {
      XTreeObject obj = (XTreeObject) menuStack.peek();
      treeModel.insertNodeInto(node, obj.getXTreeNode(), obj.getIndex());
      obj.incrementIndex();
    }
  }

  public XTree(View parent)
  {
    super();
    this.parent = parent;
    setLayout(new BorderLayout());

    /* Use default icons
    UIManager.put("Tree.expandedIcon", Utilities.getIcon("images/down_arrow.gif"));
    UIManager.put("Tree.collapsedIcon", Utilities.getIcon("images/right_arrow.gif"));
    UIManager.put("Tree.leftChildIndent", new Integer(5));
    UIManager.put("Tree.rightChildIndent", new Integer(7));
    */

    root = new XTreeNode("XInsert");
    treeModel = new DefaultTreeModel(root);
    tree = new XTreeTree(treeModel);
    ToolTipManager.sharedInstance().registerComponent(tree);
    tree.addTreeSelectionListener(this);
    tree.putClientProperty("JTree.lineStyle", "Angled");
    //tree.addMouseListener(ml);

    tree.setCellRenderer(new XTreeCellRenderer());

    init();

    //No mnemonics as they are confusing/ do not work when docked.
    JPanel pane = new JPanel(new BorderLayout());
    pane.add(collapse = new JButton(Utilities.getIcon("images/button_collapse.gif")),BorderLayout.WEST);
    collapse.setToolTipText(jEdit.getProperty("xtree.collapse.button"));
    //collapse.setMnemonic(jEdit.getProperty("xtree.collapse.mnemonic").charAt(0));
    collapse.addActionListener(this);

    pane.add(reload = new JButton(Utilities.getIcon("images/menu_reload.gif")),BorderLayout.CENTER);
    reload.setToolTipText(jEdit.getProperty("xtree.reload.button"));
    //reload.setMnemonic(jEdit.getProperty("xtree.reload.mnemonic").charAt(0));
    reload.addActionListener(this);

    pane.add(expand = new JButton(Utilities.getIcon("images/button_expand.gif")),BorderLayout.EAST);
    expand.setToolTipText(jEdit.getProperty("xtree.expand.button"));
    //expand.setMnemonic(jEdit.getProperty("xtree.expand.mnemonic").charAt(0));
    expand.addActionListener(this);

    add(pane, BorderLayout.NORTH);
    add(new JScrollPane(tree), BorderLayout.CENTER);

    JPanel optionPane = new JPanel(new BorderLayout());
    optionPane.add(carriageReturn = new JCheckBox(jEdit.getProperty("xtree.carriage.label")), BorderLayout.NORTH);
    carriageReturn.setSelected(jEdit.getBooleanProperty("xtree.carriage", false));
    carriageReturn.addActionListener(this);

    optionPane.add(executeScript = new JCheckBox(jEdit.getProperty("xtree.execute.label")), BorderLayout.CENTER);
    executeScript.setSelected(jEdit.getBooleanProperty("xtree.execute", true));
    if (jEdit.getProperty("xtree.execute") == null)
      executeScript.setSelected(true);
    executeScript.addActionListener(this);

    add(optionPane, BorderLayout.SOUTH);
  }

  private void init()
  {
    inserts = new Vector(200);
    menuStack = new Stack();
    rootIndex = 0;
    if(jEdit.getBooleanProperty("xinsert.display.all", true))
    {
      int i =0;
      String current;
      while((current = jEdit.getProperty("xinsert.inserts." + i)) != null)
      {
        loadInternalInsert(current);
        i++;
      }
    }
    //Macros menu
    if(jEdit.getBooleanProperty("xinsert.display.macros", true))
    {
      addMenu("Macros");
      Vector vec = Macros.getMacroHierarchy();
      Iterator iter = vec.iterator();
      while(iter.hasNext())
      {
        Object o = iter.next();
        if(o instanceof Vector)
        {
          loadMacroVector((Vector)o);
        }
        else if(o instanceof String)
        {
            loadNamedMacro(Macros.getMacro((String)o));
        }
        else
        {
          loadNamedMacro((Macros.Macro)o);
        }
      }
      closeMenu();
    }
    build();
    tree.expandRow(0);
    tree.setRootVisible(false);
    tree.setShowsRootHandles(true);
  }

  private void loadMacroVector(Vector v)
  {

    Iterator iter = v.iterator();
    addMenu(iter.next().toString());
    while(iter.hasNext())
    {
      Object o = iter.next();
      if(o instanceof Vector)
      {
        loadMacroVector((Vector)o);
      }
      else
      {
        loadNamedMacro(Macros.getMacro(o.toString())); //NDT fix...
      }
    }
    closeMenu();
  }

  private void loadNamedMacro(Macros.Macro macro)
  {
    addInsert(macro.getLabel(), macro.getName(), XTreeItem.NAMED_MACRO_TYPE);
  }

  private void loadInternalInsert(String fileName)
  {
    try
    {
      if(jEdit.getBooleanProperty("xinsert.display." + fileName, true))
      {
        if(!XInsertReader.read(this, XTree.class.getResourceAsStream(fileName + ".insert.xml"), fileName + ".xinsert.xml"))
          Log.log(Log.ERROR,this,("Resource not found: " + fileName));
        else
          Log.log(Log.NOTICE,this,("Resource loaded: " + fileName));
      }
    }
    catch(NullPointerException e)
    {
      Log.log(Log.ERROR,this,("Resource not found: " + fileName));
    }
  }

  private void reload()
  {
    root.removeAllChildren();
    init();
    treeModel.reload();
    tree.repaint();
  }

  private void build()
  {
    String dir = jEdit.getProperty("xinsert.inserts-directory");
    if(!dir.endsWith(File.separator))
      dir = dir + File.separator;
    File f = new File(dir);
    if (!f.exists())
      f.mkdirs();
    String inserts[] = Utilities.getWildCardMatches(dir, "*.insert.xml", false);
    if (inserts == null)
      return;

    try
    {
      String fileName;
      for (int i = 0; i < inserts.length; i++)
      {
        fileName = dir + inserts[i];
        if (XInsertReader.read(this, new FileInputStream(fileName), fileName))
        {
          String[] args = { inserts[i] };
          System.out.println(jEdit.getProperty("xtree.loaded", args));
        }
      }
    }
    catch (FileNotFoundException fnfe)
    {
    }
  }

  /*MouseListener ml = new MouseAdapter() {
    public void mouseClicked(MouseEvent e)
    {
      if(e.getClickCount() == 1 &&
        ((e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK))
        {
          Log.log(Log.DEBUG, this, "Mouse Clicked");
          XTreeNode node;
          TreePath tPath = tree.getClosestPathForLocation(e.getX(), e.getY());
          if(tPath != null)
          {
            // Prevents it registering clicks when expanding nodes and scrolling simultaneously
            Rectangle bounds = tree.getPathBounds(tPath);
            if(e.getX() < bounds.getX()-10)
              return;

              node = (XTreeNode) tPath.getLastPathComponent();
              if(!node.isLeaf()) return;
              if(node.getChildCount() != 0) return;
              if(node.getIndex() != -1)
              {
                insert(node);
              }
              parent.requestFocus();
              parent.toFront();

          }
        }

}};*/

  public void valueChanged(TreeSelectionEvent e)
  {
    JTree source = (JTree) e.getSource();
    if (source.isSelectionEmpty())
      return;
    XTreeNode node = (XTreeNode) source.getSelectionPath().getLastPathComponent();
    if (node.getIndex() != -1)
    {
      insert(node);
    }
    parent.getTextArea().requestFocus();
    parent.toFront();
    source.clearSelection();
  }

  public void reload(DefaultTreeModel model)
  {
    tree.setModel(model);
  }

  public void actionPerformed(ActionEvent evt)
  {
    Object o = evt.getSource();
    if (o == expand)
    {
      for (int i = 0; i < tree.getRowCount(); i++)
        tree.expandRow(i);
    }
    else if (o == collapse)
    {
      for (int i = tree.getRowCount(); i >= 0; i--)
        tree.collapseRow(i);
    }
    else if (o == reload)
    {
      reload();
    }
    else if (o == carriageReturn)
      jEdit.setBooleanProperty("xtree.carriage", carriageReturn.isSelected());
    else if (o == executeScript)
      jEdit.setBooleanProperty("xtree.execute", executeScript.isSelected());
  }


  private void insert(XTreeNode node)
  {
    if (!parent.getTextArea().isEditable())
    {
      parent.getToolkit().beep();
      return;
    }
    XTreeItem item = (XTreeItem) inserts.elementAt(node.getIndex()-1);
    String data = item.getContent();
    int type = item.getType();
    if(type == XTreeItem.TEXT_TYPE || !executeScript.isSelected())
      insertText(node, false);
    else if(type == XTreeItem.MACRO_TYPE)
    {
      Log.log(Log.DEBUG, this, "Running Macro...");
      XScripter.runMacro(parent, (String)node.getUserObject(), data);
      return;
    }
    else if(type == XTreeItem.XINSERT_SCRIPT_TYPE)
    {
      Log.log(Log.DEBUG, this, "Running XInsert Script ...");
      XScripter.runXInsertScript(parent, data, node);
    }
    else if(type == XTreeItem.NAMED_MACRO_TYPE)
    {
      Log.log(Log.DEBUG, this, "Running Named Macro...");
      XScripter.runNamedMacro(parent, (String)node.getUserObject(), data);
      return;
    }
    else
    {
      insertText(node, true);
    }
  }

  public void insertText(XTreeNode node, boolean error)
  {
    String data = ((XTreeItem)inserts.elementAt(node.getIndex()-1)).getContent();
    JEditTextArea textArea = parent.getTextArea();
    Buffer buffer = parent.getBuffer();
    String selected;
    buffer.beginCompoundEdit();
    if(textArea.getSelectedText() != null)
      selected = textArea.getSelectedText();
    else
      selected = "";
    int j = 0;
    int pos = -1;
    char c = '\0';
    lineNo = 0;
    StringBuffer buf = new StringBuffer(data.length());
    for (int i = 0; i < data.length(); i++)
    {
      switch((c = data.charAt(i)))
      {
      case '|':
        if (i < data.length() - 1 && data.charAt(i + 1) == '|')
        {
          i++;
          buf.append(c);
        }
        else
        {
          buf.append(selected);
          pos = j;
        }
        break;
      case '\\':
        if (i < data.length() - 1 && data.charAt(i + 1) == 'n')
        {
          i++;
          lineNo++;
          buf.append('\n');
        }
        else if (i < data.length() - 1 && data.charAt(i + 1) == 't')
        {
          i++;
          int tabSize = buffer.getTabSize();
          if(jEdit.getBooleanProperty("buffer.noTabs", false))
          {
            for(int k = 0;k<tabSize;k++)
              buf.append(" ");
          }
          else
            buf.append("\t");
        }
        else if (i < data.length() -1 && data.charAt(i + 1) == '$')
        {
          i++;
          buf.append('$');
        }
        else if (i < data.length() -1 && data.charAt(i + 1) == '\\')
        {
          i++;
          buf.append(c);
        }
        else
          buf.append(c);
        break;
      case '\n':
        lineNo++;
        buf.append(c);
        break;
      case '$':
        i++;
        int tempi = i;
        while(i < data.length() && Character.isLetterOrDigit(data.charAt(i)))
          i++;
        Log.log(Log.DEBUG, this, "$ = " + data.substring(tempi, i));
        String val = XScripter.getSubstituteFor(parent, data.substring(tempi, i), node);
        if(val == null)
          buf.append(data.substring(tempi, i));
        else
          buf.append(val);
        i--;
        break;
      default:
        buf.append(c);
        break;
      }
      j++;
    }
    int tmp = textArea.getCaretPosition();
    if (carriageReturn.isSelected())
      buf.append('\n');
    if (error)
    {
      buf.insert(0,"\nERROR: UNKNOWN INSERT TYPE\n");
    }
    textArea.setSelectedText(buf.toString());
    if (pos != -1)
      textArea.setCaretPosition(tmp + pos);
    buffer.endCompoundEdit();
  }

  private static final ImageIcon plainLeaf = Utilities.getIcon("images/tree_leaf.gif");
  private static final ImageIcon scriptLeaf = Utilities.getIcon("images/tree_leaf_script_x.gif");
  private static final ImageIcon macroLeaf = Utilities.getIcon("images/tree_leaf_macro.gif");
  private static final ImageIcon namedmacroLeaf = Utilities.getIcon("images/tree_leaf_namedmacro.gif");
  private static final ImageIcon errorLeaf = Utilities.getIcon("images/tree_leaf_error.gif");

  class XTreeCellRenderer extends DefaultTreeCellRenderer
  {
    XTreeCellRenderer()
    {
      super();

      /*openIcon = Utilities.getIcon("images/tree_open.gif");
      closedIcon = Utilities.getIcon("images/tree_close.gif");*/
      textSelectionColor = Color.red;
      borderSelectionColor = tree.getBackground();
      backgroundSelectionColor = tree.getBackground();
    }

    public Component getTreeCellRendererComponent(JTree source, Object value, boolean sel,
        boolean expanded, boolean leaf, int row,
        boolean hasFocus)
    {
      if (leaf)
      {
        TreePath path = source.getPathForRow(row);
        if (path != null)
        {
          XTreeNode node = (XTreeNode) path.getLastPathComponent();
          int index = node.getIndex();

          if (index != -1)
          {
            int type = ((XTreeItem) inserts.elementAt(index - 1)).getType();
            switch (type)
            {
            case XTreeItem.TEXT_TYPE:
              leafIcon = plainLeaf;
              break;
            case XTreeItem.XINSERT_SCRIPT_TYPE:
              leafIcon = scriptLeaf;
              break;
            case XTreeItem.MACRO_TYPE:
              leafIcon = macroLeaf;
              break;
            case XTreeItem.NAMED_MACRO_TYPE:
              leafIcon = namedmacroLeaf;
              break;
            default:
              leafIcon = errorLeaf;
            }

          }
        }
      }

      return super.getTreeCellRendererComponent(source, value, sel, expanded, leaf, row, hasFocus);
    }
  }

  private class XTreeTree extends JTree
  {
    public XTreeTree(TreeModel model)
    {
      super(model);
    }

    public String getToolTipText(MouseEvent e)
    {
      if(e == null)
        return null;
      TreePath tPath = tree.getPathForLocation(e.getX(), e.getY());
      if(tPath != null)
      {
        XTreeNode node = (XTreeNode) tPath.getLastPathComponent();
        if(!node.isLeaf())
          return null;
        try
        {
          XTreeItem item = (XTreeItem) inserts.elementAt(node.getIndex()-1);
          int type = item.getType();
          String content = item.getContent();
          if(type == XTreeItem.TEXT_TYPE)
            return content;
          else if(type == XTreeItem.MACRO_TYPE)
            return "Macro";
          else if(type == XTreeItem.XINSERT_SCRIPT_TYPE)
            return "Script";
          else if(type == XTreeItem.NAMED_MACRO_TYPE)
            return "Named Macro";
          else
            return "Error: " + content;
        }
        catch( ArrayIndexOutOfBoundsException ex)
        {
          //   Log.log(Log.ERROR, XTree.class, "getTreeToolTip() throws "
          //    + ex.getClass().getName() + " exception.");
          //   Log.log(Log.ERROR, XTree.class, "TreePath is " + tPath.toString());
          //   Log.log(Log.ERROR, XTree.class, "TreeNode object is " +
          //    node.toString());
          return null;
        }
      }
      else
        return null;
    }
  }

}

// End of XTree.java
